#!/usr/bin/env bb
(ns switch-workspace
  (:require [babashka.process :as p]
            [clojure.string :as s]
            [cheshire.core :as j]))

(defn niri-msg
  "Generic function for `niri msg action`"
  [action params]
  (let [cmd (into ["niri" "msg" "--json" (name action)] params)]
    (as-> cmd $
        (apply p/shell {:out :string :err :string} $)
        (select-keys $ [:out :err]))))

(defn focused-or-nil?
  "Return if a workspaces is focused or doesn't have a name."
  [workspace]
  (let [{:keys [name is_focused]} workspace]
    (or (nil? name)
        (true? is_focused))))

(defn niri-get-workspaces
  "Get all the workspaces from niri"
  []
  (as-> (niri-msg :workspaces nil) $
       (:out $)
       (j/parse-string $ keyword)
       (remove focused-or-nil? $)
       (map :name $)))

(defn niri-focus-workspace
  "Set focus to the given namespace"
  [workspace]
  (niri-msg :action ["focus-workspace" workspace]))

(defn run-rofi
  "Given a list of options, call rofi and return the selected one."
  [options]
  (as-> options $
    (s/join "\n" $)
    (p/shell {:out :string :in $} "rofi" "-dmenu")
    (:out $)
    (s/trim-newline $)))

(defn -main []
  (-> (niri-get-workspaces)
      (run-rofi)
      (niri-focus-workspace)))

(-main)
