#!/usr/bin/env bb

(ns arrange-workspaces
  (:require [babashka.process :as p]
            [clojure.edn :as edn]))

(defonce config-path
  (str (System/getenv "HOME")
       "/.config/kanshi/workspaces.edn"))

(defn read-config
  "Read the configuration from `config-path`"
   [path]
  (-> path
      slurp
      edn/read-string))

(defn niri-msg
  "Generic function for `niri msg action`"
  [action params]
  (let [cmd (into ["niri" "msg" "--json" "action" (name action)] params)]
    (->> cmd
         (apply p/shell {:out :string :err :string})
         (select-keys [:out :err]))))

(defn niri-move-to-monitor
  "Move `workspace` to `monitor`."
  [monitor workspace]
  (niri-msg :move-workspace-to-monitor ["--reference" workspace monitor]))

(defn arrange [context]
  (doseq [[monitor workspaces] context]
    (let [f (partial niri-move-to-monitor monitor)]
      (mapv f workspaces))))

(defn -main [args]
  (try
    (let [config  (read-config config-path)
          context (first args)]
      (when-let [context-config (get config context nil)]
        (println "Found" context "configuration")
        (arrange context-config)))
    (catch java.io.FileNotFoundException _
      (.println System/err (str "Configuration " config-path " not found."))
      (System/exit -1))))

(-main *command-line-args*)
